<?php

/**
 * 2007-2019 PrestaShop SA and Contributors
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Academic Free License (AFL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * https://opensource.org/licenses/AFL-3.0
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade PrestaShop to newer
 * versions in the future. If you wish to customize PrestaShop for your
 * needs please refer to https://www.prestashop.com for more information.
 *
 * @author    PrestaShop SA <contact@prestashop.com>
 * @copyright 2007-2019 PrestaShop SA and Contributors
 * @license   https://opensource.org/licenses/AFL-3.0  Academic Free License (AFL 3.0)
 * International Registered Trademark & Property of PrestaShop SA
 */

class LinkifyIntegrationModuleFrontController extends ModuleFrontController
{

    public function initContent(){
        parent::initContent();
        $this->handleGetRequest();
    }
    
    //##########################################################
    //######## HANDLE GET AND POST REQUESTS ####################
    //##########################################################

    public function postProcess(){
        //In the dev server the GET requests are processed here, not in the initContent(). So we add this patch
        if($_SERVER['REQUEST_METHOD'] == 'GET'){
            $this->handleGetRequest();
        }

        $request_body = json_decode(file_get_contents('php://input'), true);
        if ($_SERVER['REQUEST_METHOD'] == 'POST'){
            if ($request_body["action"] == "notification"){
                $this->handlePostRequest($request_body);
            }elseif ($request_body["action"] == "cancellation"){
                $this->handleDeleteRequest($request_body);
            }else{
                $this->sendHttpResponse(['message'=> 'Invalid action'], 400);
            }
        }
        else{
            $this->sendHttpResponse(['message'=> $_SERVER['REQUEST_METHOD'].' not allowed'], 400);
        } 
    }

    private function handleGetRequest(){
        if($_SERVER['REQUEST_METHOD'] != 'GET'){
            return;
        }
        $required_params = ['id'];
        $request_params = $this->getRequiredParamsFromGET($required_params);
        $this->handleBadRequest($required_params,$request_params);
        $cart_id = $request_params['id'];        
        $cart = new Cart($cart_id);
        if(!$cart || $cart->id == null){
            $this->sendHttpResponse(['message'=> "No se encontró el carro #{$cart_id}"], 400);
        }
        $currency = new CurrencyCore($cart->id_currency);
        if($currency->iso_code != "CLP") {
            $this->sendHttpResponse(['message' => "El carro de compras no está en CLP"] , 422);
        }

        $pending_amount = $cart->getOrderTotal(true, Cart::BOTH) - $this->getCartPaidAmount($cart_id);
        if($pending_amount <= 0) {
            $this->sendHttpResponse(['message' => "El carro de compras no está pendiente de pago"] , 422);
        }

        $this->sendHttpResponse([
            'amount' => $pending_amount, 
            'description' => $this->buildCartDescription($cart), 
            'currency' => $currency->iso_code
        ] ,200);
    }

    private function handlePostRequest($request_body){

        $this->handleBadRequest(['id', 'transfers', 'completeness', 'original_amount', 'currency'], $request_body);
        $cart_id = $request_body['id'];     
        $transfers = $request_body['transfers'];
        $completeness = $request_body['completeness'];
        $original_amount = $request_body['original_amount'];

        $cart = new Cart($cart_id);
        if(!$cart || $cart->id == null){
            $this->sendHttpResponse(['message'=> "No se encontró el carro #{$cart_id}"], 400);
        }
        $currency = new CurrencyCore($cart->id_currency);
        if($currency->iso_code != $request_body['currency']){
            $this->sendHttpResponse([
                'status' => 'rejected', 
                'message'=> 'La moneda del carro de compra no coincide con la validada por Linkify'
            ], 200);
        }

        if($cart->OrderExists()){
            $this->sendHttpResponse(['message'=> 'Ya existe una orden activa para este carro de compras'], 404);
        }

        $cart_total = $cart->getOrderTotal(true, Cart::BOTH);
        $linkify_payment = new Linkify();

        // This consider cases when this is not the first time the notification happens (previous failed when some transactions where already added to DB)
        $unnotified_transfers = [];
        $notified_transfers_amount = 0;
        //Search for transfers in this notification that are already added to the DB
        foreach($transfers as $transfer){
            $ps_payment = $this->getPrestashopLinkedPayment($transfer["hashid"]);
            if($ps_payment){
                // If is linked to this cart, omit
                if($ps_payment["prestashop_cart_id"] == $cart_id){
                    $notified_transfers_amount += $transfer["amount"];
                    continue;
                }
                // If it is linked to another cart, return error
                else{
                    $this->sendHttpResponse([
                        'status' => 'rejected',
                        'message' => "Error: La transferencia #{$transfer["hashid"]} está vinculada a otro carro"
                    ],200);
                }
            }else{
                $unnotified_transfers[] = $transfer;
            }
        }

        if($request_body['original_amount'] - $notified_transfers_amount != ($cart_total - $this->getCartPaidAmount($cart_id))){
            $this->sendHttpResponse([
                'status'  => 'rejected',
                'message' => "El carro #$cart_id fue modificada por lo que no se pudo procesar el pago. Reinicie el proceso de validación para cargar la nueva información de la orden."
            ],200);
        }

        if(sizeof($unnotified_transfers) > 0){
            foreach($unnotified_transfers as $unnotified_transfer){
                $this->addTransferToDb($cart_id, $unnotified_transfer["hashid"], $unnotified_transfer["amount"]);
            }
        }

        if ($completeness == 'underpaid'){
            $checkout_message = 'Pago parcial procesado correctamente. Aun quedan $'.($cart_total - $this->getCartPaidAmount($cart_id)). ' por pagar. Reinicie el proceso para pagar el restante';
            $this->sendHttpResponse([
                'status'  => 'accepted',
                'message' => $checkout_message, 
                'restart' => true
            ], 200);
        }else{
            $linkify_payment->validateOrder(
                $cart->id, 
                (int)Configuration::get('PS_OS_PAYMENT'), 
                $cart_total, //We put the exact amount (and not max($cart_total, $transfers_total_amount) because prestashop shows them as payment error if not exact)
                'Linkify'
            );

            $this->sendHttpResponse([
                'status'   => 'accepted',
                'redirect' => $this->getPrestashopThankYouPageUrl($cart_id), 
                'message'  => 'Pago recibido correctamente', 
                'restart'  => false
            ], 200);
        }
    }

    private function handleDeleteRequest($request_body){
        // If any transfer is not linked to an order, return error
        $cart_id = $request_body['id'];
        $transfers = $request_body['transfers'];
        foreach($transfers as $transfer){
            $ps_payment = $this->getPrestashopLinkedPayment($transfer["hashid"]);
            if($ps_payment){
                // If is linked to this order, omit
                if($ps_payment["prestashop_cart_id"] == $cart_id){
                    continue;
                }
                //If linked to other order, error
                else{
                    $this->sendHttpResponse([
                            "message" => "Error: Transferencia #{$transfer["hashid"]} está vinculada a otro carro", 
                            "notify_merchant" => false
                    ],400);
                }
            }
            //If not linked to any order, error
            else{
                $this->sendHttpResponse([
                        "message" => "Error: Transferencia #{$transfer["hashid"]} no está vinculada al carro", 
                        "notify_merchant" => false
                ],400);
            }
        }

        foreach($transfers as $transfer){
            $this->unlinkTransferFromCart($cart_id, $transfer["hashid"]);
        }
        $this->sendHttpResponse(['message' => 'Transferencia(s) desvinculada(s). Recuerde que en Prestashop debe revertir manualmente el pago'],200);
    }
    

    //##########################################################
    //################## HELPER METHODS ########################
    //##########################################################

    private function getPrestashopThankYouPageUrl($cart_id){
        $order = new Order(Order::getOrderByCartId($cart_id));
        $customer = $order->getCustomer();
        $linkify_module_id = Module::getInstanceByName($order->module)->id;
        return Tools::getShopDomainSsl(true,true) . __PS_BASE_URI__ . 'index.php?controller=order-confirmation&id_cart=' . $cart_id
                . '&id_module=' . (int)$linkify_module_id . '&id_order=' . $order->id . '&key=' . $customer->secure_key;
    }

    private function buildCartDescription($cart){
        $order_detail_string = '';
        foreach($cart->getProducts() as $product){
            $product_name = $product['name']; //Product name asociated to the item
            $quantity_sold = $product['cart_quantity']; //How many of that product was sold
            $pluralize = $quantity_sold == 1 ? "" : "es";
            $order_detail_string = $order_detail_string ."<strong>" . $product_name . "</strong> - " . $quantity_sold . " unidad" . $pluralize . "\n";
        }
        return $order_detail_string;
    }

    //The function should send a response using a json codification of that array
    private function sendHttpResponse($response_array, $status_code){
        header('HTTP/1.1 ' . $status_code . ' OK');
        header('Content-Type: application/json');
        die(Tools::jsonEncode($response_array));
    }

    //If the request does not meet the expected body fields, answer with error
    private function handleBadRequest($expected_body_fields, $actual_body){
        //Check that the request comes from Linkify
        if(!array_key_exists('HTTP_X_LINKIFY_CONFIRMATION',$_SERVER)){
            $this->sendHttpResponse(['message'=> 'Token de autenticación no enviado'], 400);
        }
        $linkify_confirmation_token = $_SERVER['HTTP_X_LINKIFY_CONFIRMATION']; 
        $this->checkHmac($linkify_confirmation_token,json_encode($actual_body));
        
        //Check that all the expected parameters are received
        foreach($expected_body_fields as $expected_field){
            if(!array_key_exists($expected_field, $actual_body)){
                $this->sendHttpResponse(['message'=> 'Missing body param'], 400);
            }
        }

        return true;
    }

    //Check if a request is from Linkify
    private function checkHmac($incoming_hmac, $json_encoded_body){
        $secret_key = Configuration::get('LINKIFY_SECRET_KEY');
        $computed_hmac = hash_hmac('sha256',$json_encoded_body,$secret_key, false);
        if($computed_hmac != $incoming_hmac){
            $this->sendHttpResponse(['message'=> 'Firma inválida.'], 400);
        }
    }

    private function getRequiredParamsFromGET($required_params){
        $return_array = [];
        $get_data = json_decode($_GET["encoded_data"], true);
        foreach($required_params as $key){
            if(isset($get_data[$key])){
                $return_array[$key] = $get_data[$key];
            }
        }
        return $return_array;
    }


    #Database functions
    private function getCartPaidAmount($cart_id){
        $transactions_summary = Db::getInstance()->getRow('SELECT SUM(amount) as amount FROM ' . _DB_PREFIX_ . 'linkify_transfers WHERE prestashop_cart_id = ' . pSQL($cart_id) . ' AND linked = 1');
        return intval($transactions_summary["amount"]);
    }

    private function addTransferToDB($cart_id, $linkify_transfer_hashid, $amount){
        Db::getInstance()->execute('INSERT INTO ' . _DB_PREFIX_ . 'linkify_transfers (linked, prestashop_cart_id, linkify_transfer_hashid, amount, creation_date, last_update_date) VALUES (1, ' . pSQL($cart_id) . ', \'' . pSQL($linkify_transfer_hashid) . '\', ' . pSQL($amount) . ', NOW(), NOW())');
    }

    private function unlinkTransferFromCart($cart_id, $linkify_transfer_hashid){
        Db::getInstance()->execute('UPDATE ' . _DB_PREFIX_ . 'linkify_transfers SET linked = NULL, last_update_date = NOW() WHERE prestashop_cart_id = ' . pSQL($cart_id) . ' AND linkify_transfer_hashid = \'' . pSQL($linkify_transfer_hashid) . '\'');
    }

    private function getPrestashopLinkedPayment($linkify_transfer_hashid){
        return Db::getInstance()->getRow('SELECT * FROM ' . _DB_PREFIX_ . 'linkify_transfers WHERE linkify_transfer_hashid = \'' . pSQL($linkify_transfer_hashid) . '\' AND linked = 1');
    }

}
