<?php

/**
 * 2007-2019 PrestaShop SA and Contributors
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Academic Free License (AFL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * https://opensource.org/licenses/AFL-3.0
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade PrestaShop to newer
 * versions in the future. If you wish to customize PrestaShop for your
 * needs please refer to https://www.prestashop.com for more information.
 *
 * @author    PrestaShop SA <contact@prestashop.com>
 * @copyright 2007-2019 PrestaShop SA and Contributors
 * @license   https://opensource.org/licenses/AFL-3.0  Academic Free License (AFL 3.0)
 * International Registered Trademark & Property of PrestaShop SA
 */

use PrestaShop\PrestaShop\Core\Payment\PaymentOption;

if (!defined('_PS_VERSION_')) {
    exit;
}

class Linkify extends PaymentModule
{
    public function __construct()
    {
        $this->name = 'linkify';
        $this->tab = 'payments_gateways';
        $this->version = '1.0.0';
        $this->author = 'Linkify';
        $this->need_instance = 0;
        $this->ps_versions_compliancy = [
            'min' => '1.6',
            'max' => _PS_VERSION_
        ];
        $this->bootstrap = true;
        $this->currencies = true;
        $this->currencies_mode = 'checkbox';
        parent::__construct();

        $this->displayName = $this->l('Linkify');
        $this->description = $this->l('Valida transferencias electrónicas en Chile de manera automática.');
        if (!Configuration::get('LINKIFY_MERCHANT_ID')) {
            $this->warning = $this->l('No se entregaron credenciales válidas.');
        }
    }

    public function install(){
        return (
            parent::install() 
            && $this->registerHook('paymentOptions') 
            && $this->registerHook('paymentReturn') 
            && $this->installLinkifyTables()
            && Configuration::updateValue('LINKIFY_MERCHANT_ID', '') 
            && Configuration::updateValue('LINKIFY_SECRET_KEY', '')
        );
    }

    public function uninstall(){
            return (
                parent::uninstall() 
                && Configuration::deleteByName('LINKIFY_MERCHANT_ID')
                && Configuration::deleteByName('LINKIFY_SECRET_KEY')
                && $this->unregisterHook('paymentOptions')
                && $this->unregisterHook('paymentReturn') 
            );
    }

    public function installLinkifyTables() {
        // Run sql for creating DB tables
        Db::getInstance()->execute(
            'CREATE TABLE IF NOT EXISTS ' . _DB_PREFIX_ . 'linkify_transfers ('.
            'linked TINYINT(1) DEFAULT NULL COMMENT "NULL when transfer has been unlinked from the cart. 1 when transfer is assigned to the cart", '.
            'linkify_transfer_hashid VARCHAR(255) CHARACTER SET utf8mb4 COLLATE utf8mb4_bin NOT NULL, '.
            'prestashop_cart_id INT(11) NOT NULL, '.
            'amount INT(11) NOT NULL, '.
            'creation_date DATETIME NOT NULL, '.
            'last_update_date DATETIME NOT NULL, '.
            'UNIQUE KEY (linked, linkify_transfer_hashid))'
        ); 
    }

    public function hookPaymentOptions($params){
        $payment_options = [ $this->getExternalPaymentOption()];
        return $payment_options;
    }

    public function getExternalPaymentOption(){
        $externalOption = new PaymentOption();
        $externalOption -> setCallToActionText('Trasferencia electrónica - Validada por Linkify')
                        -> setAction($this->context->link->getModuleLink($this->name, 'create', [], true))
                        -> setLogo('https://www.linkify.cl/images/logos/mini_isotype.png')
                        ;
        return $externalOption;
    }
    
    //Returns true if there is an error with provided credentials. To be used in form to avoid saving of invalid data.
    private function checkLinkifyCredentialErrors($merchant_id, $secret_key){
        $endpoint = 'https://app.linkify.cl/api/check_credentials/' . $merchant_id;
        
        $integration_url = Tools::getShopDomainSsl(true,true) . __PS_BASE_URI__ . 'index.php?fc=module&module=linkify&controller=integration';
        $params = ['secret_key' => $secret_key, 'integration_url'=> $integration_url];
        $url = $endpoint . '?' . http_build_query($params);
        $ch = curl_init();
        curl_setopt($ch,CURLOPT_CUSTOMREQUEST,'GET');
        curl_setopt($ch,CURLOPT_URL,$url);
        curl_setopt($ch,CURLOPT_RETURNTRANSFER,true);
        curl_setopt($ch,CURLOPT_HTTPHEADER,['Content-Type: application/json']);
        $response = json_decode(curl_exec($ch));
        $http_code = curl_getinfo($ch,CURLINFO_HTTP_CODE);
        if($http_code === 200 && $response->success){//No error
            return false;
        }
        return true; //Error
    }

    public function getContent()
    {
        $output = null;

        if (Tools::isSubmit('submit'.$this->name)) {
            $merchant_id = strval(Tools::getValue('LINKIFY_MERCHANT_ID'));
            $secret_key = strval(Tools::getValue('LINKIFY_SECRET_KEY'));

            if ($this->checkLinkifyCredentialErrors($merchant_id,$secret_key)) {
                $output .= $this->displayError($this->l('Invalid Configuration value'));
            } else {
                Configuration::updateValue('LINKIFY_MERCHANT_ID', $merchant_id);
                Configuration::updateValue('LINKIFY_SECRET_KEY', $secret_key);
                $output .= $this->displayConfirmation($this->l('Settings updated'));
            }
        }
        return $output.$this->displayForm();
    }

    private function displayForm(){
        // Get default language
        $defaultLang = (int)Configuration::get('PS_LANG_DEFAULT');

        // Init Fields form array
        $fieldsForm[0]['form'] = [
            'legend' => [
                'title' => $this->l('Settings'),
            ],
            'input' => [
                [
                    'type' => 'text',
                    'label' => $this->l('ID de cuenta Linkify'),
                    'name' => 'LINKIFY_MERCHANT_ID',
                    'size' => 20,
                    'required' => true
                ],
                [
                    'type' => 'text',
                    'label' => $this->l('Clave privada Linkify'),
                    'name' => 'LINKIFY_SECRET_KEY',
                    'size' => 20,
                    'required' => true
                ]
            ],
            'submit' => [
                'title' => $this->l('Save'),
                'class' => 'btn btn-default pull-right'
            ]
        ];
    
        $helper = new HelperForm();
    
        // Module, token and currentIndex
        $helper->module = $this;
        $helper->name_controller = $this->name;
        $helper->token = Tools::getAdminTokenLite('AdminModules');
        $helper->currentIndex = AdminController::$currentIndex.'&configure='.$this->name;
    
        // Language
        $helper->default_form_language = $defaultLang;
        $helper->allow_employee_form_lang = $defaultLang;
    
        // Title and toolbar
        $helper->title = $this->displayName;
        $helper->show_toolbar = true;        // false -> remove toolbar
        $helper->toolbar_scroll = true;      // yes - > Toolbar is always visible on the top of the screen.
        $helper->submit_action = 'submit'.$this->name;
        $helper->toolbar_btn = [
            'save' => [
                'desc' => $this->l('Save'),
                'href' => AdminController::$currentIndex.'&configure='.$this->name.'&save'.$this->name.
                '&token='.Tools::getAdminTokenLite('AdminModules'),
            ],
            'back' => [
                'href' => AdminController::$currentIndex.'&token='.Tools::getAdminTokenLite('AdminModules'),
                'desc' => $this->l('Back to list')
            ]
        ];
    
        // Load current value
        $helper->fields_value['LINKIFY_MERCHANT_ID'] = Configuration::get('LINKIFY_MERCHANT_ID');
        $helper->fields_value['LINKIFY_SECRET_KEY'] = Configuration::get('LINKIFY_SECRET_KEY');

        return $helper->generateForm($fieldsForm);
    }
}
